<?php

namespace IgniteKit\WP\Notices;

/**
 * Class Notice
 * @package IgniteKit\WP\Notices
 */
class Notice implements NoticesInterface {

	/**
	 * The unique ID (autogenerated)
	 * @var string
	 */
	public $id;

	/**
	 * The key
	 *
	 * @var string
	 */
	public $key;

	/**
	 * The type
	 *
	 * @var string
	 */
	public $type;

	/**
	 * The message
	 *
	 * @var message
	 */
	public $message;

	/**
	 * The dismissal expiration time
	 *
	 * Eg. 'disabled', 'never', or number of seconds.
	 * @var string|int
	 */
	public $expiry;

	/**
	 * The dismiss url
	 * @var string
	 */
	public $dismiss_url;

	/**
	 * The prefix
	 * @var string
	 */
	public $prefix;

	/**
	 * Notice constructor.
	 *
	 * @param $key
	 * @param $type
	 * @param $message
	 * @param string $expiry
	 * @param string $dismiss_url
	 * @param $prefix
	 */
	public function __construct( $key, $type, $message, $expiry = 'never', $dismiss_url = '', $prefix = '' ) {
		$this->id          = self::generate_id( $key, $type );
		$this->key         = $key;
		$this->type        = $type;
		$this->message     = $message;
		$this->expiry      = $expiry;
		$this->prefix      = $prefix;
		$this->dismiss_url = add_query_arg( array( 'notice_id' => $this->id ), $dismiss_url );
	}

	/**
	 * Dismisses the notice
	 */
	public function dismiss() {
		if ( $this->expiry === self::DISMISS_FOREVER ) {
			update_option( $this->id, 'yes' );
		} else if ( is_numeric( $this->expiry ) ) {
			set_transient( $this->id, 'yes', (int) $this->expiry );
		}
	}


	/**
	 * Removes notice dismissal flag. After this call the notice is not dismissed anymore.
	 */
	public function reset() {
		if ( $this->expiry === self::DISMISS_FOREVER ) {
			delete_option( $this->id );
		} else if ( is_numeric( $this->expiry ) ) {
			delete_transient( $this->id );
		}
	}

	/**
	 * Check if notice is dismissed.
	 *
	 * @return bool
	 */
	public function is_dismissed() {
		if ( $this->expiry === self::DISMISS_FOREVER ) {
			return get_option( $this->id ) === 'yes';
		} else if ( is_numeric( $this->expiry ) ) {
			return get_transient( $this->id ) === 'yes';
		} else {
			return false;
		}
	}

	/**
	 * Prints notice
	 */
	public function print() {

		if ( $this->is_dismissed() ) {
			return;
		}

		$dismiss_url = add_query_arg( '_wpnonce', wp_create_nonce( $this->get_nonce_key() ), $this->dismiss_url );

		$dismissable = $this->expiry !== 'disabled' ? 'is-dismissible' : '';
		$class       = sprintf( 'notice notice-%s %s dg-notice dg-notice-%s', $this->type, $dismissable, $this->prefix );
		printf( '<div class="%1$s" data-dismiss-url="%2$s">%3$s</div>', esc_attr( $class ), $dismiss_url, $this->message );
	}

	/**
	 * The nonce key
	 * @return string
	 */
	private function get_nonce_key() {
		return sprintf( '%s_nonce', $this->prefix );
	}

	/**
	 * Generate notice id
	 *
	 * @param $key
	 * @param $type
	 *
	 * @return string
	 */
	public static function generate_id( $key, $type ) {
		return sprintf( 'nc_%s_%s', $type, $key );
	}
}